//******************************************************************************
//  MSP430 AD9851 DDS VCO
//******************************************************************************

#include "msp430.h"

#define CLOCK	0x08	// W_CLK bit
#define LOAD	0x02	// FQ_UD bit
#define DATA	0x20	// D7 bit
#define RESET	0x04

void delay();
void pulse_clock();
void pulse_load();
void send_byte(unsigned char b);
void send_word(unsigned int w);
void compute_tw();
unsigned long compute_pi(float mhz);
void send_tw();

struct tune{
	union{
		unsigned long	pi;
		unsigned int	piw[2];
		unsigned char	pib[4];
		}p;
	unsigned char	po;
	}t;

unsigned long freq, base, incr, ofs;


int n=0;

int main(void)
	{
	int i;
	
	WDTCTL = WDTPW + WDTHOLD;		// Stop watchdog timer
	// set clock to 16MHz
  	BCSCTL1 = CALBC1_16MHZ;
	DCOCTL = CALDCO_16MHZ;
	P1DIR |= 0x2f;					// Set P1.0,1,3,5 to output direction
	P1OUT = 0x2f;					// set all bits to 1, for inverting drivers
									// bit 2 is RESET, non-inverting
	// setup A/D converter
	ADC10CTL0 = ADC10SHT_2 +ADC10ON;			// S/H time = 16 clocks
	ADC10CTL1 = INCH_4;				// select analog input A4
	ADC10AE0 |= 0x10;				// PA.4 ADC option select
	

	// wait before accessing AD9851
	for(i=0; i<100; i++)
		delay();
	
	// deassert RESET
	P1OUT &= ~RESET;

	// enable serial load on AD9851
	pulse_clock();
	pulse_load();
	
	P1OUT &= ~DATA;
	for(i=0; i<45; i++)	// load all zeros
		pulse_clock();
	pulse_load();
		
	t.po=1;	// set phase offset=0, mpy on
	
	// set base frequency
	base=compute_pi(1.0);
		
	for (;;){
		unsigned int v;

    		P1OUT ^= 0x01;				// flash the LED
		

		// sample input
		ADC10CTL0 |= ENC + ADC10SC;   // Sampling and conversion start
		while(ADC10CTL1 & ADC10BUSY)  // wait for conversion to complete
			;
		v=ADC10MEM & 0x3ff;
		incr = v;
		incr <<= 10;
		ofs=0x200<<10;

		// set the frequency
		t.p.pi=base+incr-ofs;

		
		send_tw();
  		}
	}

#define SET_BITS 0x50
#define CLEAR_BITS 0xaf

void
pulse_clock()
	{
	P1OUT &= ~CLOCK;
	P1OUT |= CLOCK;
	}

void
pulse_load()
	{
	P1OUT &= ~LOAD;
	P1OUT |= LOAD;
	}

unsigned long
compute_pi(float mhz)
	{
	double p;

	p=4294967296.0*(mhz/180.0);
	return((unsigned long)p);
	}

void
send_tw()
	{

	
	// load the programming word
		send_word(t.p.piw[0]);
		send_word(t.p.piw[1]);

		send_byte(t.po);
		pulse_load();
	}

void
send_byte(unsigned char b)
	{
	register int i;
	
	for(i=0; i<8; i++){
		if(b&1)
			P1OUT &= ~DATA;
		else
			P1OUT |= DATA;

		//pulse_clock
		P1OUT &= ~CLOCK;
		P1OUT |= CLOCK;

		b >>= 1;
		}
	}

void
send_word(unsigned int w)
	{
	register int i;
	
	for(i=16; i!=0; i--){
		if(w&1)
			P1OUT &= ~DATA;
		else
			P1OUT |= DATA;

		P1OUT &= ~CLOCK;
		P1OUT |= CLOCK;

		w >>= 1;
		}
	}
	
void delay()
	{
	volatile unsigned int i=77;
	
	do{i--;} while(i!=0);
	}
